<?php
namespace BK;

use BK\PDO\SQLite;

class Record {
    public $table;
    public $params;
    public $id = 'id';
    public $columns = [];
    public $passwordFields = [];
    public $listPageColumns = [];
    public $types = [];
    public $record = [];
    public $exists = false;
    public $db;

    /**
     * Record constructor.
     * @param SQLite $db
     * @param null $id
     */
    function __construct(SQLite $db, $id = null) {
        $this->db = $db;
        if ($id) {
            $this->load($id);
        }
        if (!isset($this->columns[$this->table])) {
            $this->columns[$this->table] = $this->db->getColumns($this->table);
        }
    }

    public function getTable($table) {
        return $this->db->queryAll("SELECT * FROM `{$table}`");
    }

    public function load($id) {
        $this->record = $this->get($id);
        $this->exists = $this->exists();
        if (!$this->exists && $id != 'new' && $id > 0) {
            throw new \Exception("Record does not exist");
        }
        return $this->record;
    }

    public function exists(){
        return (empty($this->record) ? false : true);
    }

    public function clean($params) {
        if (!isset($this->columns[$this->table])) {
            $this->columns[$this->table] = $this->db->getColumns($this->table);
        }
        if (!isset($this->types[$this->table])) {
            $this->types[$this->table] = $this->db->getColumnTypes($this->table);
        }
        $cleaned = [];
        foreach ($params as $field => $value) {
            if (in_array($field, $this->columns[$this->table])) {
//                if (strstr($this->types[$this->table][$field]['data_type'], 'date')) {
//                    $value = $this->utcDate($value);
//                }
                $cleaned[$field] = $value;
            }
        }
        return $cleaned;
    }

    public function save($params = null) {
        if (!$params && !empty($this->record)) {
            $params = $this->record;
        }
        if (!$params) {
            return false;
        }
        if (isset($_SESSION['user'])) {
            $params['user_id'] = $_SESSION['user']['id'];
        }
        $params = $this->clean($params);
        $fields = [];
        $values = [];
        $updates = [];
        $placeholders = [];
        if ((!isset($params[$this->id]) || $params[$this->id] == 'new') && !$this->record) {
//            echo "Create!\n";
//            print_r(($this->columns[$this->table]));
            $params[$this->id] = null;
            if (isset($this->columns[$this->table]['created_on'])) {
                $params['created_on'] = time();
            }
        }
        if (!isset($params[$this->id]) && $this->record) {
            $params[$this->id] = $this->record[$this->id];
        }

        if (isset($this->columns[$this->table]['updated_on'])) {
            $params['updated_on'] = time();
        }

        foreach ($params as $field => $value) {
            $placeholders[] = (in_array($field, $this->passwordFields) ? 'PASSWORD(?)' : '?');
            $fields[] = "`{$field}`";
            $updates[] = "`{$field}` = VALUES(`{$field}`)";
            $values[] = $value;
        }
        $sql = "
            REPLACE INTO `{$this->table}` 
              (" . implode(', ', $fields) . ") 
            VALUES (" . implode(', ', $placeholders) . ") 
            ";
//        echo "<pre>";
//        echo $sql;
//        print_r($values);
//        die();
        try {
            $prep = $this->db->prepare($sql);
        } catch (\PDOException $e) {
            throw new \Exception("[" . __METHOD__ . "] Prepare: " . $e->getMessage());
        }
        try {
            $prep->execute($values);
        } catch (\PDOException $e) {
            throw new \Exception("[" . __METHOD__ . "] Execute: " . $e->getMessage());
        }
        try {
            $recordid = $this->db->lastInsertId();
        } catch (\PDOException $e) { }

        if (isset($params[$this->id])) {
            $id = $params[$this->id];
        } else {
            $id = $recordid;
        }
//        var_dump($id);
//        print_r($this->get($id));die();
        return $id;
    }

    public function get($id) {
        if (!$id) {
            return false;
        }
        $prep = $this->db->prepare("SELECT * FROM `{$this->table}` WHERE `{$this->id}` = ?");
        return $prep->execute([$id])->fetchRow();
    }

    public function delete($id) {
        $prep = $this->db->prepare("DELETE FROM `{$this->table}` WHERE `{$this->id}` = ?");
        $result = $prep->execute([$id]);

        return $result;

    }

    public function listAll($where = "", $values = []) {
        $where = $where == "" ? "" : "AND " . $where;
        $sql = "SELECT * FROM `{$this->table}` {$where}";
        $prep = $this->db->prepare($sql);
        return $prep->execute($values)->fetchAll();
    }

    public function listActive() {
        if (!isset($this->columns[$this->table])) {
            $this->columns[$this->table] = $this->db->getColumns($this->table);
        }
        $where = '';
        if (isset($this->columns[$this->table]['active'])) {
            $where = " WHERE active = 1";
        }
        $sql = "SELECT * FROM `{$this->table}`" . $where;
        $prep = $this->db->prepare($sql);
        return $prep->execute([])->fetchAll();
    }

    public function listByKey($key = 'id', $value='name', $where = '') {
        if (!isset($this->columns[$this->table])) {
            $this->columns[$this->table] = $this->db->getColumns($this->table);
        }
        $sql = "SELECT * FROM `{$this->table}` {$where}";
        $prep = $this->db->prepare($sql);
        $rows = $prep->execute([])->fetchAll();
        return $this->recordsByKey($key, $value, $rows);
    }

    public function recordsByKey($key = 'id', $value = 'name', $rows = []) {
        $results = [];
        foreach ($rows as $row) {
            $resultKey = '';
            if (is_array($key)) {
                $string = '';
                foreach ($key as $col) {
                    if (!isset($this->columns[$this->table][$col])) {
                        $string .= $col;
                    } else {
                        $string .= $row[$col];
                    }
                }
                $resultKey = $string;
            } else {
                $resultKey = $row[$key];
            }
            if (is_array($value)) {
                $string = '';
                foreach ($value as $col) {
                    if (!isset($this->columns[$this->table][$col])) {
                        $string .= $col;
                    } else {
                        $string .= $row[$col];
                    }
                }
                $results[$resultKey] = $string;
            } else {
                $results[$resultKey] = $row[$value];
            }
        }
        return $results;
    }

    public function listPageData($data = [], $columns = []) {
        if (!isset($data['limit']) || empty($data['limit'])) {
            $data['limit'] = 10;
        }
        if (!isset($data['page']) || empty($data['limit'])) {
            $data['page'] = 1;
        }
        if (!isset($data['offset']) || empty($data['offset'])) {
            $data['offset'] = ($data['limit'] * ($data['page'] - 1));;
        }
        if (!isset($data['search']) || empty($data['search'])) {
            $data['search'] = '';
        }
        if (!isset($data['equal']) || empty($data['equal'])) {
            $data['equal'] = [];
        }
        if (!isset($data['not']) || empty($data['not'])) {
            $data['not'] = [];
        }
        if (!isset($data['in']) || empty($data['in'])) {
            $data['in'] = [];
        }
        if (!isset($data['query']) || empty($data['query'])) {
            $data['query'] = [];
        }
        if (!isset($data['order']) || empty($data['order'])) {
            $data['order'] = [];
        }
        if (empty($this->listPageColumns) && !empty($columns)) {
            $this->listPageColumns = $columns;
        }
        $search = '';
        $values = [];
        if (trim($data['search']) != '') {
            $where = [];
            foreach ($this->listPageColumns as $col) {
                $where[] = "{$col} LIKE ?";
                $values[] = "%{$data['search']}%";
            }
            $search = 'WHERE (' . implode(' OR ', $where) . ')';
        }
        if (!empty($data['equal'])) {
            $where = [];
            foreach ($data['equal'] as $column => $value) {
                $where[] = "`{$column}` = ?";
                $values[] = $value;
            }

            if ($search == '') {
                $search = "WHERE " . implode(' AND ', $where);
            } else {
                $search .= " AND " . implode(' AND ', $where);
            }
        }
        if (!empty($data['not'])) {
            $where = [];
            foreach ($data['not'] as $column => $value) {
                $where[] = "`{$column}` != ?";
                $values[] = $value;
            }

            if ($search == '') {
                $search = "WHERE " . implode(' AND ', $where);
            } else {
                $search .= " AND " . implode(' AND ', $where);
            }
        }
        if (!empty($data['query'])) {
            if ($search == '') {
                $search = "WHERE " . implode(' ', $data['query']);
            } else {
                $search .= " AND " . implode(' ', $data['query']);
            }
        }
        if (!empty($data['in'])) {
            if ($search == '') {
                $search = "WHERE " . "{$data['in']['field']} IN ('" . implode("','", $data['in']['values']) . "')";
            } else {
                $search .= " " . "{$data['in']['field']} IN ('" . implode("','", $data['in']['values']) . "')";
            }
        }
        $order = '';
        if (!empty($data['order'])) {
            $order = 'ORDER BY '.implode(', ', $data['order']);
        }
        if (in_array('access', $this->columns[$this->table])) {
            $access = (!isset($_SESSION['user']) ? 0 : $_SESSION['user']['access']);
            if ($search == '') {
                $search = "WHERE";
            } else {
                $search .= " AND";
            }
            $search .= " `access` <= {$access}";
        }
        $sql = "
            SELECT
              COUNT(*)
            FROM
              `{$this->table}`
            {$search}     
        ";
        $total = $this->db->prepare($sql)->execute($values)->fetchOne();
        if ($data['limit'] == 'All') {
            $data['offset'] = 0;
            $data['limit'] = $total;
        }
        $sql = "
            SELECT 
              * 
            FROM 
              `{$this->table}`
            {$search}
                 
            {$order}
            LIMIT {$data['offset']}, {$data['limit']}
        ";
//        echo "<pre>";
//        print_r($sql);
//        print_r($values); die();
        $records = $this->db->prepare($sql)->execute($values)->fetchAll();
        return [
            'columns' => $this->listPageColumns,
            'records' => $records,
            'page_current' => ($data['offset'] == 0 ? 1 : ($data['offset'] / $data['limit']) + 1),
            'page_total' => ($total == 0 ? 1 : ceil($total / $data['limit'])),
            'rec_total' => $total,
            'offset' => $data['offset'],
            'limit' => ($data['limit'] >= $total ? 'All' : $data['limit']),
            'search' => $data['search']
        ];
    }
}